"""Service handling authentication and basic RBAC."""
from __future__ import annotations

from typing import Optional, Tuple

# Local imports (no package prefix)
from repositories.user_repo import UserRepository
from utils.hashing import verify_password, hash_password
from models.user import User, Role


class AuthService:
    def __init__(self, repo: Optional[UserRepository] = None) -> None:
        self.repo = repo or UserRepository()
        # Ensure demo users exist if seed was not run
        self._ensure_seed_users()

    def _ensure_seed_users(self) -> None:
        """Create default users if they do not exist (safe to run repeatedly)."""
        defaults = [
            ("admin", "Admin@123", Role.ADMIN),
            ("manager", "Manager@123", Role.MANAGER),
            ("staff", "Staff@123", Role.STAFF),
        ]
        for username, pwd, role in defaults:
            existing = self.repo.get_user_by_username(username)
            if not existing:
                self.repo.create_user(
                    User(username=username, hashed_password=hash_password(pwd), role=role)
                )

    def login(self, username: str, password: str) -> Tuple[bool, Optional[dict]]:
        """
        Validate credentials.
        Returns (True, user_dict) on success, else (False, None).
        """
        if not username or not password:
            return False, None

        user = self.repo.get_user_by_username(username)
        if not user:
            return False, None

        if verify_password(password, user.get("hashed_password", "")):
            return True, user

        return False, None
